package com.ving.gasmileage;

import android.content.Context;
import android.content.SharedPreferences;
import android.net.Uri;
import android.os.Handler;
import android.os.Looper;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

public class InitData {

    // 1. Define a callback interface to notify the MainActivity when work is done.
    public interface OnInitCompleteListener {
        void onInitComplete();
        void onInitError(String errorMessage);
    }

    // --- These are the ONLY member variables you need ---
    private final Context mContext;
    private final SharedPreferences settings;
    private final ExecutorService executorService;
    private final Handler mainHandler;
    private OnInitCompleteListener mListener;
    private String task;
    private String errorMsg;
    private boolean shouldShowAddCarDialog = false;

    InitData(MainActivity act, SharedPreferences sharedPreferences) {
        this.mContext = act;
        this.settings = sharedPreferences;
        this.executorService = Executors.newSingleThreadExecutor();
        this.mainHandler = new Handler(Looper.getMainLooper());
    }

    public void noAddCarDialogNeeded() {
        shouldShowAddCarDialog = false;
    }

    public boolean shouldShowAddCarDialog() {
        return shouldShowAddCarDialog;
    }

    // Call this from MainActivity to set the listener
    public void setOnInitCompleteListener(OnInitCompleteListener listener) {
        this.mListener = listener;
    }

    // Method to initiate the initialization process
    public void startInitialization(final MyApplication myApp, final Uri selectedDirectoryUri) {
        // Guard clause: If there's no directory selected, don't start the background work.
        if (selectedDirectoryUri == null) {
            if (mListener != null) {
                mListener.onInitComplete(); // Tell MainActivity we are "done"
            }
            return;
        }

        myApp.performingInitTask();
        executorService.execute(() -> {
            try {
                String carInfoData = settings.getString("CarInfo", "");
                String csvFile = null;
                if (carInfoData != null && carInfoData != "" && !carInfoData.isEmpty()) {
                    myApp.carInfo().save(carInfoData);
                    String currentCar = settings.getString("CurrentCar", "");
                    myApp.carInfo().setCurrentCar(currentCar);
                    csvFile = myApp.carInfo().csvFile();
                }
                myApp.setFileState(false);
                if (csvFile != null && !csvFile.isEmpty()) {
                    ReadMileageData rmd = new ReadMileageData(mContext, selectedDirectoryUri, csvFile, false);
                    if (!rmd.setUpData()) {
                        task = "Finding CSV file in directory";
                        errorMsg = rmd.error();
                        postError(task + "\n" + errorMsg);
                        return;
                    } else {
                        if (!rmd.readFile()) {
                            task = "Reading the CSV file";
                            errorMsg = rmd.error();
                            postError(task + "\n" + errorMsg);
                            return;
                        } else {
                            // If read is successful, load the data into the app
                            for (MileageData md : rmd.getMileageData()) {
                                myApp.addData(md);
                            }
                            myApp.setFileState(true);
                            myApp.setChanged(false);
                        }
                    }
                }
            } catch (Exception e) {
                // Catch any unexpected errors during background processing
                postError("An unexpected error occurred during initialization: " + e.getMessage());
            }
            postSuccess(myApp);
        });
    }

    private void postSuccess(final MyApplication myApp) {
        mainHandler.post(() -> {
            // Check if a car needs to be created
            String csvFile = myApp.carInfo().csvFile();
            if (csvFile == null || csvFile.isEmpty()) {
                shouldShowAddCarDialog = true;
            }
            if (mListener != null) {
                mListener.onInitComplete();
            }
        });
    }

    private void postError(final String errorMsg) {
        mainHandler.post(() -> {
            // Notify the MainActivity of the error
            if (mListener != null) {
                mListener.onInitError(errorMsg);
            }
        });
    }
}
